// Perception.h: interface for the CPerception class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PERCEPTION_H
#define PERCEPTION_H

#pragma warning (disable : 4786)

#include <vector>
#include <string>
#include <queue>
#include "GameState.h"
#include "PerceptionTypes.h"
#include "SystemClock.h"

using namespace std;

/**
* Basic class for storing general information about seen objects
* Notes: if m_tSetValues is SV_None, than are the information stored in this or derived classes invalid
* @author frz 
* @version 24.2.2001
*/
class CSeenObject {
    friend class CPerception;
protected:
    double m_dDist;
    double m_dDir;
    double m_dDistChng;
    double m_dDirChng;
    int m_tSetValues; // SV_Dist = 8, SV_Dir = 4, SV_DistChng = 2, SV_DirChng = 1

protected:
    CSeenObject(double dist, double dir, double distChng, double dirChng)
    {
       m_tSetValues = SEE_DISTANCE | SEE_ANGLE | SEE_DISTANCE_CHANGE | SEE_ANGLE_CHANGE;
       m_dDist = dist;
       m_dDir = dir;
       m_dDistChng = distChng;
       m_dDirChng = dirChng;
    }
    void SetDistance(double newDist) {m_dDist = newDist; m_tSetValues |= SEE_DISTANCE;}
    void SetDirection(double newDir) {m_dDir = newDir; m_tSetValues |= SEE_ANGLE ;}
    void SetDistanceChng(double newDistChng) {m_dDistChng = newDistChng; m_tSetValues |= SEE_DISTANCE_CHANGE;}
    void SetDirectionChng(double newDirChng) {m_dDirChng = newDirChng; m_tSetValues |= SEE_ANGLE_CHANGE;}
    void SetValuesVar(int val) {m_tSetValues = val;}
    void Invalidate() {m_tSetValues = SEE_NOTHING;}
public:
    CSeenObject() {m_tSetValues = SEE_NOTHING;}
	double GetDistance() const {return m_dDist;}
	double GetDirection() const {return m_dDir;}
	double GetDistanceChng() const {return m_dDistChng;}
	double GetDirectionChng() const {return m_dDirChng;}
	int GetSetValuesVar() const {return m_tSetValues;}
    void operator=(CSeenObject seenObject) {m_tSetValues = seenObject.m_tSetValues; m_dDist = seenObject.m_dDist; m_dDir = seenObject.m_dDir; m_dDistChng = seenObject.m_dDistChng; m_dDirChng = seenObject.m_dDirChng;}
	bool IsValid() const{return (m_tSetValues != SEE_NOTHING);} // return true if the informations stored in class are valid
};

/**
* Storing information about seen object "flag"
* @author frz 
* @version 24.2.2001
*/
class CSeenFlag : public CSeenObject {
    friend class CPerception;
private:
    EFlagType m_tFlagID;

    CSeenFlag(EFlagType flagID) {m_tFlagID = flagID;}
    void SetFlagID(EFlagType newFlagID) {m_tFlagID = newFlagID;}
public:
    CSeenFlag() {m_tFlagID = FT_NoFlag;}
    EFlagType GetFlagID() {return m_tFlagID;}
    void operator=(CSeenFlag seenObject) {m_tSetValues = seenObject.m_tSetValues; m_dDist = seenObject.m_dDist; m_dDir = seenObject.m_dDir; m_dDistChng = seenObject.m_dDistChng; m_dDirChng = seenObject.m_dDirChng; m_tFlagID = seenObject.m_tFlagID;}
};

/**
* Storing information about seen object "goal"
* @author frz 
* @version 24.2.2001
*/
class CSeenGoal : public CSeenObject {
    friend class CPerception;
private:
    EGoalType m_tGoalID;

    CSeenGoal(EGoalType goalID) {m_tGoalID = goalID;}
    void SetGoalID(EGoalType newGoalID) {m_tGoalID = newGoalID;}
public:
    CSeenGoal() {m_tGoalID = GT_NoGoal;}
    EGoalType GetGoalID() {return m_tGoalID;}
    void operator=(CSeenGoal seenObject) {m_tSetValues = seenObject.m_tSetValues; m_dDist = seenObject.m_dDist; m_dDir = seenObject.m_dDir; m_dDistChng = seenObject.m_dDistChng; m_dDirChng = seenObject.m_dDirChng; m_tGoalID = seenObject.m_tGoalID;}
};

/**
* Storing information about seen object "line"
* @author frz 
* @version 24.2.2001
*/
class CSeenLine : public CSeenObject {
    friend class CPerception;
private:
    ELineType m_tLineID;

    CSeenLine(ELineType lineID) {m_tLineID = lineID;}
    void SetLineID(ELineType newLineID) {m_tLineID = newLineID;}
public:
    CSeenLine() {m_tLineID = LT_NoLine;}
    ELineType GetLineID() {return m_tLineID;}
    void operator=(CSeenLine seenObject) {m_tSetValues = seenObject.m_tSetValues; m_dDist = seenObject.m_dDist; m_dDir = seenObject.m_dDir; m_dDistChng = seenObject.m_dDistChng; m_dDirChng = seenObject.m_dDirChng; m_tLineID = seenObject.m_tLineID;}
};

/**
* Storing information about seen object "player"
* Notes: if uniform number is -1 and team name is "", then this attributes were not in message from server (player was too far or view quality is low)
* @author frz 
* @version 24.2.2001 
*/
class CSeenPlayer : public CSeenObject {
    friend class CPerception;
private:
    double m_dBodyDir;
    double m_dHeadDir;
    char m_szTeamName[16];
    int m_nUNum;

    CSeenPlayer(int uNum, char* teamName, double bodyDir, double headDir) {m_nUNum = uNum; strcpy(m_szTeamName, teamName); m_dBodyDir = bodyDir; m_dHeadDir = headDir;}
    void SetTeamName(char* teamName) {strcpy(m_szTeamName, teamName);}
    void SetUniformNum(int newUNum) {m_nUNum = newUNum;}
    void SetBodyDir(double newBodyDir) {m_dBodyDir = newBodyDir;}
    void SetHeadDir(double newHeadDir) {m_dHeadDir = newHeadDir;}
public:
    bool GetSide() const {return (strcmp("tim4",m_szTeamName)==0);}  // FIXME
    CSeenPlayer() {m_nUNum = -1, m_szTeamName[0] = '\0';}
    void GetTeamName(char* teamName) {strcpy(teamName, m_szTeamName);}
    int GetUniformNum() const {return m_nUNum;}
    int GetBodyDir() const {return m_dBodyDir;}
    int GetHeadDir() const {return m_dHeadDir;}
    void operator=(CSeenPlayer seenObject) {m_tSetValues = seenObject.m_tSetValues; m_dDist = seenObject.m_dDist; m_dDir = seenObject.m_dDir; m_dDistChng = seenObject.m_dDistChng; m_dDirChng = seenObject.m_dDirChng; m_nUNum = seenObject.m_nUNum; strcpy(m_szTeamName, seenObject.m_szTeamName); m_dBodyDir = seenObject.m_dBodyDir; m_dHeadDir = seenObject.m_dHeadDir;}
};

/**
* Storing information about "hear" message
* Notes: if m_tMsgType is MT_None, than the information stored in this class are invalid
* @author frz 
* @version 7.12.2000
*/
class CHeardMsg {
    friend class CPerception;
private:
    string m_sMessage;
    EHearMsgType m_tMsgType; // MT_Nobody, MT_Referee, MT_Self, MT_Other
    EPlayMode m_tPlayMode;
    double m_dDir;
private:
    CHeardMsg(double dir, EHearMsgType type, char* msg) {m_dDir = dir; m_tMsgType = type; m_sMessage = msg;}
    CHeardMsg(double dir, EHearMsgType type, string msg) {m_dDir = dir; m_tMsgType = type; m_sMessage = msg;}
    void SetMsgType(EHearMsgType type) {m_tMsgType = type;}
    void SetDir(double dir) {m_dDir = dir;}
    void SetMessage(string msg) {m_sMessage = msg;}
    void SetMessage(char* msg) {m_sMessage = msg;}
    void SetPlayMode(EPlayMode tPlayMode) {m_tPlayMode = tPlayMode;}
    void Invalidate() {m_tMsgType = MT_Nobody;}
public:
    CHeardMsg() {m_tMsgType = MT_Nobody;}
    EHearMsgType GetMsgType() {return m_tMsgType;}
    bool GetMessage(char* msg) {strcpy(msg, m_sMessage.data()); return (m_tMsgType != MT_Nobody);}
    bool GetMessage(string* msg) {*msg = m_sMessage; return (m_tMsgType != MT_Nobody);}
    double GetDir() {return m_dDir;}
    bool IsValid() {return m_tMsgType != MT_Nobody;} // return true if the information is valid
    EPlayMode GetPlayMode() {return m_tPlayMode;}
};

/**
* Storing information about "sense body" message
* Notes: if  is , than the information stored in this class are invalid
* @author frz 
* @version 7.12.2000
*/
class CSenseBody {
    friend class CPerception;
private:
    CSenseBody(EViewQuality tViewQuality, EViewWidth tViewWidth, double dStamina, double dEffort, double dSpeed, double dSpeedDir, double dHeadDir, int nKickCount, int nDashCount, int nTurnCount, int nSayCount, int nHeadTurnCount)
	{
		m_bValid = true;
		m_tViewQuality = tViewQuality;
		m_tViewWidth = tViewWidth;
		m_dStamina = dStamina; m_dEffort = dEffort;
		m_dSpeed = dSpeed;
		m_dSpeedDir = dSpeedDir;
		m_dHeadDir = dHeadDir;
		m_nKickCount = nKickCount;
		m_nDashCount = nDashCount;
		m_nTurnCount = nTurnCount;
		m_nSayCount = nSayCount;
		m_nHeadTurnCount = nHeadTurnCount;
	}//For FRZ : I kill you, if you change this 14 lines back to one !!!
    EViewQuality m_tViewQuality; // VQ_None, VQ_Low, VQ_High
    EViewWidth m_tViewWidth; // VW_None, VW_Narrow, VW_Normal, VW_Wide
    bool m_bValid;
    double m_dStamina;
    double m_dEffort;
    double m_dSpeed,m_dSpeedDir;
    double m_dHeadDir;
    int m_nKickCount;
    int m_nDashCount;
    int m_nTurnCount;
    int m_nSayCount;
    int m_nHeadTurnCount;
    void SetViewQuality(EViewQuality tViewQuality) {m_tViewQuality = tViewQuality;}
    void SetViewWidth(EViewWidth tViewWidth) {m_tViewWidth = tViewWidth;}
    void SetStamina(double dStamina) {m_dStamina = dStamina;}
    void SetEffort(double dEffort) {m_dEffort = dEffort;}
    void SetSpeed(double dSpeed,double dSpeedDir) {m_dSpeed = dSpeed;m_dSpeedDir=dSpeedDir;}
    void SetHeadDir(double dHeadDir) {m_dHeadDir = dHeadDir;}
    void SetKickCount(int nKickCount) {m_nKickCount = nKickCount;}
    void SetDashCount(int nDashCount) {m_nDashCount = nDashCount;}
    void SetTurnCount(int nTurnCount) {m_nTurnCount = nTurnCount;}
    void SetSayCount(int nSayCount) {m_nSayCount = nSayCount;}
    void SetHeadTurnCount(int nHeadTurnCount) {m_nHeadTurnCount = nHeadTurnCount;}
    void SetAsValid() {m_bValid = true;}
    void Invalidate() {m_bValid = false;}
public:
    CSenseBody() {m_bValid = false;}
    EViewQuality GetViewQuality() {return m_tViewQuality;}
    EViewWidth GetViewWidth() {return m_tViewWidth;}
    double GetStamina() {return m_dStamina;}
    double GetEffort() {return m_dEffort;}
    double GetSpeed() {return m_dSpeed;}
    double GetSpeedDir() {return m_dSpeedDir;}
    double GetHeadDir() {return m_dHeadDir;}
    int GetKickCount() {return m_nKickCount;}
    int GetDashCount() {return m_nDashCount;}
    int GetTurnCount() {return m_nTurnCount;}
    int GetSayCount() {return m_nSayCount;}
    int GetHeadTurnCount() {return m_nHeadTurnCount;}
    void operator=(CSenseBody senseBody) {m_bValid = senseBody.IsValid(); m_tViewQuality = senseBody.GetViewQuality(); m_tViewWidth = senseBody.GetViewWidth(); m_dStamina = senseBody.GetStamina(); m_dEffort = senseBody.GetEffort(); m_dSpeed = senseBody.GetSpeed(); m_dHeadDir = senseBody.GetHeadDir(); m_nKickCount = senseBody.GetKickCount(); m_nDashCount = senseBody.GetDashCount(); m_nTurnCount = senseBody.GetTurnCount(); m_nSayCount = senseBody.GetSayCount(); m_nHeadTurnCount = senseBody.GetHeadTurnCount();}
    bool IsValid() {return m_bValid;}
};

/**
* Storing information about "init" message
* Notes: if  is , than the information stored in this class are invalid
* @author frz 
* @version 7.12.2000
*/
class CInitMsg
{
    friend class CPerception;
private:
    CInitMsg(int nUniformNum, ESide tSide, EPlayMode tPlayMode) {m_bValid = true; m_nUniformNum = nUniformNum; m_tSide = tSide; m_tPlayMode = tPlayMode;}
    bool m_bValid;
    int m_nUniformNum;
    ESide m_tSide;
    EPlayMode m_tPlayMode;
    void SetUniformNum(int nUniformNum) {m_nUniformNum = nUniformNum;}
    void SetSide(ESide tSide) {m_tSide = tSide;}
    void SetPlayMode(EPlayMode tPlayMode) {m_tPlayMode = tPlayMode;}
    void SetAsValid() {m_bValid = true;}
    void Invalidate() {m_bValid = false;}
public:
    CInitMsg() {m_bValid = false;}
    int GetUniformNum() {return m_nUniformNum;}
    ESide GetSide() {return m_tSide;}
    EPlayMode GetPlayMode() {return m_tPlayMode;}
    void operator=(CInitMsg init) {m_bValid = init.IsValid(); m_nUniformNum = init.GetUniformNum(); m_tSide = init.GetSide(); m_tPlayMode = init.GetPlayMode();}
    bool IsValid() {return m_bValid;}
};

/* type definition for vectors used for storing information about seen objects */
typedef vector<CSeenPlayer> CSeenPlayersVect;
typedef vector<CSeenFlag> CSeenFlagsVect;
typedef vector<CSeenGoal> CSeenGoalsVect;
typedef vector<CSeenLine> CSeenLinesVect;
typedef CSeenPlayersVect::iterator CSeenPlayersIter;
typedef CSeenFlagsVect::iterator CSeenFlagsIter;
typedef CSeenGoalsVect::iterator CSeenGoalsIter;
typedef CSeenLinesVect::iterator CSeenLinesIter;

/**
* Agregation of all objects (seen objects, heard messeges, init mesage), whitch were created during parsing message from server
* Notes: if  is , than the information stored in this class are invalid
* @author frz 
* @version 24.2.2001
*/
class CPerception
{
    friend class CServerMsgTranslator;

/* private attributes*/
private:
    /* general perception's data */
    TSystemTime m_nReceivedTime;
    int m_nPlayTime;
    CGameTime m_tGameTime;
    EMsgType m_tType;

    /* data for message see */
    int m_nSeenPlayersCount;
    int m_nSeenPlaymatesCount;
    int m_nSeenFlagsCount;
    int m_nSeenLinesCount;
    bool m_bSeeLeftGoal;
    bool m_bSeeRightGoal;
    bool m_bSeeBall;

    /* vectors for seen object */
    CSeenPlayersVect m_cSeenPlayers;
    CSeenFlagsVect m_cSeenFlags;
    CSeenGoalsVect m_cSeenGoals;
    CSeenLinesVect m_cSeenLines;

private:
    CSeenObject m_cBall;
    CHeardMsg m_cHeardMsg; // class for message hear
    CSenseBody m_cSenseBody;  // class for sense body message
    CInitMsg m_cInitMsg;  // class for init message


private:
    CPerception(int playTime, CGameTime gameTime, EMsgType mgsType);  // consructor - used by class CParser

    void SetMsgType(EMsgType mMsgType) {m_tType = mMsgType;}
    /* adding new seen objects in perception */
    void AddSeenPlayer(int uNum, char* teamName, double dist, double dir, double distChng, double dirChng, int setValues, double bodyDir, double headDir);
    void AddSeenFlag(EFlagType flagID, double dist, double dir, double distChng, double dirChng, int setValues);
    void AddSeenLine(ELineType lineID, double dist, double dir, double distChng, double dirChng, int setValues);
    void AddSeenGoal(EGoalType goalID, double dist, double dir, double distChng, double dirChng, int setValues);
    void AddSeenBall(double dist, double dir, double distChng, double dirChng, int setValues);
    /* setting information from hear message */
    void SetHeardMsg(double dir, EHearMsgType type, EPlayMode tPlayMode, char* msg) {m_cHeardMsg.SetDir(dir); m_cHeardMsg.SetMsgType(type); m_cHeardMsg.SetPlayMode(tPlayMode); m_cHeardMsg.SetMessage(msg);}
    /* setting information from sense body message */
    void SetSenseBody(EViewQuality tViewQuality, EViewWidth tViewWidth, double dStamina, double dEffort, double dSpeed ,double dSpeedDir, double dHeadDir, int nKickCount, int nDashCount, int nTurnCount, int nSayCount, int nHeadTurnCount) 
	{
		m_cSenseBody.SetAsValid();
		m_cSenseBody.SetViewQuality(tViewQuality);
		m_cSenseBody.SetViewWidth(tViewWidth);
		m_cSenseBody.SetStamina(dStamina);
		m_cSenseBody.SetEffort(dEffort);
		m_cSenseBody.SetSpeed(dSpeed,dSpeedDir);
		m_cSenseBody.SetHeadDir(dHeadDir);
		m_cSenseBody.SetKickCount(nKickCount);
		m_cSenseBody.SetDashCount(nDashCount);
		m_cSenseBody.SetTurnCount(nTurnCount);
		m_cSenseBody.SetSayCount(nSayCount);
		m_cSenseBody.SetHeadTurnCount(nHeadTurnCount);
	} //For FRZ : I kill you, if you change this 14 lines back to one !!!
    /* setting information from sense body message */
    void SetInitMsg(int nUniformNum, ESide tSide, EPlayMode tPlayMode) {m_cInitMsg.SetAsValid(); m_cInitMsg.SetUniformNum(nUniformNum); m_cInitMsg.SetSide(tSide); m_cInitMsg.SetPlayMode(tPlayMode);}

public:
    /* clear the content of perception */
    void ClearAll();

public:
    /* constuctor and access methods for private data of class perception */
    CPerception();
    ~CPerception();

    int GetPlayTime() {return m_nPlayTime;}
    void SetPlayTime(int nPlayTime) {m_nPlayTime = nPlayTime;}

    CGameTime GetGameTime() {return m_tGameTime;}
    void SetGameTime(CGameTime tGameTime) {m_tGameTime = tGameTime;}

    EPlayMode GetPlayMode();
//    void SetPlayMode(EPlayMode mPlayMode) { m_mPlayMode = mPlayMode; }

    TSystemTime GetReceivedTime() {return m_nReceivedTime;}
    void SetReceivedTime(int nReceivedTime) {m_nReceivedTime = nReceivedTime;}

    EMsgType GetMsgType() {return m_tType;}

    /* access methods for seen objects */
    int GetPlayersCount() {return m_nSeenPlayersCount;}
    int GetPlaymatesCount() {return m_nSeenPlaymatesCount;}
    int GetSeenFlagsCount() {return m_nSeenFlagsCount;}
    int GetSeenLinesCount() {return m_nSeenLinesCount;}
    bool IsLeftGoalSeen() {return m_bSeeLeftGoal;}
    bool IsRightGoalSeen() {return m_bSeeRightGoal;}
    bool IsBallSeen() {return m_bSeeBall;}
    /**
    *Iterator for seen players vector
    *@return iterator object that points at the first element of the sequence or just beyond the end of an empty sequence
    */
    CSeenPlayersIter GetPlayersIter() {return m_cSeenPlayers.begin();}
    /**
    *Iterator for seen flags vector
    *@return iterator object that points at the first element of the sequence or just beyond the end of an empty sequence
    */
    CSeenFlagsIter GetFlagsIter() {return m_cSeenFlags.begin();}
    /**
    *Iterator for seen goals vector
    *@return iterator object that points at the first element of the sequence or just beyond the end of an empty sequence
    */
    CSeenGoalsIter GetGoalsIter() {return m_cSeenGoals.begin();}
    /**
    *Iterator for seen lines vector
    *@return iterator object that points at the first element of the sequence or just beyond the end of an empty sequence
    */
    CSeenLinesIter GetLinesIter() {return m_cSeenLines.begin();}

    /* other iterators for seen perception's objects stored in vectors */
    int GetFirstPlayer(CSeenPlayer* player);
    int GetFirstFlag(CSeenFlag* flag);
    int GetFirstGoal(CSeenGoal* goal);
    int GetFirstLine(CSeenLine* line);
    int GetNextPlayer(CSeenPlayer* player, int currPos);
    int GetNextFlag(CSeenFlag* flag, int currPos);
    int GetNextGoal(CSeenGoal* goal, int currPos);
    int GetNextLine(CSeenLine* line, int currPos);

    /* access methods for vectors of seen perception's objects */
    bool GetBall(CSeenObject* ball);
    CSeenPlayersVect GetSeenPlayers() {return m_cSeenPlayers;}
    CSeenFlagsVect GetSeenFlags() {return m_cSeenFlags;}
    CSeenGoalsVect GetSeenGoals() {return m_cSeenGoals;}
    CSeenLinesVect GetSeenLines() {return m_cSeenLines;}
    /* access method for hear object */
    void GetHeardMgs(CHeardMsg* heardMsg) {heardMsg->SetDir(m_cHeardMsg.m_dDir); heardMsg->SetMsgType(m_cHeardMsg.m_tMsgType); heardMsg->SetMessage(m_cHeardMsg.m_sMessage);}
    /* access method for sense body object */
    void GetSenseBody(CSenseBody* senseBody);
    /* access method for sense body object */
    void GetInitMsg(CInitMsg* init) {if(m_cInitMsg.IsValid()) init->SetAsValid(); else init->Invalidate(); init->SetUniformNum(m_cInitMsg.GetUniformNum()); init->SetSide(m_cInitMsg.GetSide()); init->SetPlayMode(m_cInitMsg.GetPlayMode());}

    void operator=(CPerception perception);
};

typedef queue<CPerception*> CPerceptionQueue;


/**
* Container class for perception objects 
* @author frz
* @version 25.2.2001
*/
class CPerceptionSeq
{
private:
    CPerceptionQueue m_cPerceptions;
public:
    CPerceptionSeq() {};
    ~CPerceptionSeq() { Clear(); }

    CPerceptionSeq* AddPerception(CPerception *pPerception);
    bool GetPerception(CPerception **ppPerception);
    int GetPerceptionsCount();
    void Clear();
};

#endif  // PERCEPTIONTYPES_H
