// Geometry.cpp: implementation of the Geometry class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Player.h"
#include "Geometry.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


AngleDeg GetNormalizeAngleDeg(AngleDeg ang){
  if (fabs(ang) > 5000)
    //my_error("Huge angle passed to GetNormalizeAngleDeg");
  while (ang > 180) ang-=360;
  while (ang < -180) ang+=360;
  return ang;
}

/*********************************************************************************
 * RRectangle
 * by: jeeff
 */


RRectangle::RRectangle()
{
  left_x = right_x = top_y = bottom_y = 0;
}

RRectangle::RRectangle(const double l, const double r, const double t, const double b)
{
  //if ( l >= r ) my_error("width must be positive");
  //if ( t >= b ) my_error("height must be positive");

  left_x = l ; right_x = r ;
  top_y = t ; bottom_y = b ;
}

RRectangle::RRectangle(const Vector center,const Vector size)
{
  left_x = center.x - size.x/2.0 ;
  right_x = center.x + size.x/2.0 ;
  top_y = center.y - size.y/2.0 ;
  bottom_y = center.y + size.y/2.0 ;
}

bool RRectangle::IsWithin(const Vector &p)
{
  return (bool) ((p.x >= left_x) && (p.x <= right_x) && (p.y >= top_y) && (p.y <= bottom_y)) ;
}

Vector RRectangle::AdjustToWithin(const Vector &p){

  Vector r = p;

  if (r.y - top_y    <0) r.y = top_y;
  if (bottom_y - r.y <0) r.y = bottom_y;
  if (right_x - r.x  <0) r.x = right_x;
  if (r.x - left_x   <0) r.x = left_x;

  return r;
}

//order: top, right, bot, left
Line RRectangle::GetEdge(int n)
{
  switch (n % 4) {
  case 0: return TopEdge();
  case 1: return RightEdge();
  case 2: return BottomEdge();
  case 3: return LeftEdge();
  }
  //my_error("RRectangle::GetEdge: how did I get here");
  return Line();
}

//order: TL, TR, BR, BL
Vector RRectangle::GetPoint(int n)
{
  switch (n % 4) {
  case 0: return TopLeftCorner();
  case 1: return TopRightCorner();
  case 2: return BottomRightCorner();
  case 3: return BottomLeftCorner();
  }
  //my_error("RRectangle::GetPoint: how did I get here");
  return Vector(0,0);
}

Vector RRectangle::nearestHEdge(const Vector &p)
     /* find nearest horizontal line */
{
  static Vector r ;

  r.x = min(max(p.x,left_x),right_x) ;
  r.y = ((p.y - top_y) < (bottom_y - p.y)) ? top_y : bottom_y ;

  return r ;
}

Vector RRectangle::nearestVEdge(const Vector &p)
     /* find nearest vertical line */
{
  static Vector r ;

  r.x = ((p.x - left_x) < (right_x - p.x)) ? left_x : right_x ;
  r.y = min(max(p.y,top_y),bottom_y) ;

  return r ;
}

Vector RRectangle::nearestEdge(const Vector &p)
{
  if(min((p.x-left_x),(right_x-p.x)) < min((p.y-top_y),(bottom_y-p.y)))
    return nearestVEdge(p) ;
  else
    return nearestHEdge(p) ;
}

Line RRectangle::nearestHEdgeLine(const Vector& p)
{
  return ((p.y - top_y) < (bottom_y - p.y)) ? TopEdge() : BottomEdge() ;
}

Line RRectangle::nearestVEdgeLine(const Vector& p)
{
  return ((p.x - left_x) < (right_x - p.x)) ? LeftEdge() : RightEdge() ;
}

Line RRectangle::nearestEdgeLine(const Vector& p)
{
  if(min((p.x-left_x),(right_x-p.x)) < min((p.y-top_y),(bottom_y-p.y)))
    return nearestVEdgeLine(p) ;
  else
    return nearestHEdgeLine(p) ;
}

double RRectangle::DistanceToEdge(const Vector &p)
{
  if ( !IsWithin(p) ){
    Vector q = AdjustToWithin(p);
    return -(q.dist(p)); /* distance outside is a negative number */
  }

  return min((p.x-left_x),min((right_x-p.x),min((p.y-top_y),(bottom_y-p.y))));
}

RRectangle RRectangle::expand(double val)
{
  return RRectangle(left_x - val, right_x + val, top_y - val, bottom_y + val);
}

void  RRectangle::Print()
{
  printf("RRectangle:  x = %.1f to %.1f   y = %.1f to %.1f\n",
    LeftX(),RightX(),TopY(),BottomY());
}

Vector RRectangle::RayIntersection(Ray r)
{
  if (!IsWithin(r.origin)) {
    //my_error("RRectangle/Ray intersection only handles rays inside the RRectangle!");
    return 0;
  }

  Vector int_pt[2];
  int num_int = 0;
  for (int i=0; i < 4; i++) {
    Vector pt;
    if (GetEdge(i).RayIntersection(r, &pt))
      int_pt[num_int++] = pt;
  }

  if (num_int == 0) {
    //my_error("RRectangle ray intersection: no edge intersection?");
    return 0;
  } else if (num_int == 1) {
    return int_pt[0]; // same slope as one pair of edges
  } else if (num_int == 2) {
    RRectangle temp_rect = expand(double_EPS);
    for (int j = 0; j < 2; j++)
      if (temp_rect.IsWithin(int_pt[j]))
   return int_pt[j];
    //my_error("RRectangle ray intersection: how did I get here");
    return 0;
  } else {
    //my_error("RRectangle ray intersection: how did num_int get so big: %d", num_int);
    return 0;
  }
}


/*********************************************************************************
 * line
 * by: jeeff
 */

Line::Line(double x_coef, double y_coef, double constant)
{
  A = x_coef;
  B = y_coef;
  C = constant;
}

void Line::LineFromTwoPoints(Vector pt1, Vector pt2)
{
  double temp = (pt2.x - pt1.x);
  if (fabs(temp) < double_EPS) {
    if (fabs(pt2.y - pt1.y) < double_EPS)
      //my_error("LineFromTwoPoints: points can not be the same!");
    A = 1;
    B = 0;
  } else {
    double m = (pt2.y - pt1.y) / temp;
    A = -m;
    B = 1;
  }
  C = -(A * pt2.x + B * pt2.y);
}

void Line::LineFromRay(Ray r)
{
  if (fabs(r.direction.y) < double_EPS && fabs(r.direction.x) < double_EPS)
    //my_error("LineFromRay: dir can not be zero");
  LineFromTwoPoints(r.origin, r.origin + r.direction);
}

bool Line::PointOnLine(double x, double y)
{
  return (bool)(fabs(A * x + B * y + C) < double_EPS);
}

double Line::dist(Vector pt)
{
  return fabs( (A*pt.x + B*pt.y + C) / sqrt(sqrt(A) + sqrt(B)) );
}

double Line::dist2(Vector pt)
{
  return fabs( sqrt(A*pt.x + B*pt.y + C) / (sqrt(A) + sqrt(B)) );
}

double Line::angle()
{
  return atan(-A / B);
}

Vector Line::ProjectPointUsingCircle(Vector pt)
{
  /* here's the idea:
     first get the plane equation for the ray
     then use the closest distance formula to get the closest distance
     then using the eq for that line and the eq for a circle of the dist radius
       around our curretn position, find the nearest point.
       Whew! */

  /* compute the dist */
  Vector retPt;
  double d = dist(pt); /* the min distance */

  /* intersect the circle and the line */
  double a,b,c; /* coefficent in quadratic to solve for x */
  double disc; /* discriminant in quadratic equation */
  a = 1 + sqrt(A);
  b = 2 * (A * (pt.y + C) - pt.x);
  c = sqrt(pt.x) + sqrt(pt.y + C) - sqrt(d);
  disc = sqrt(b) - 4 * a * c;
  /* the discriminant should be zero since this is the radius
     is the closest distance between the center and the line */
  if (fabs(disc) > double_EPS)
    fprintf(stderr, "GetClosestPointToBallPath: discrimannt is bad! %f\n", disc);
  retPt.x = - b / (2 * a);

  /* we compute two possible solutions for y and then see which one is on the
     line of the ray's path */
  double sol1, sol2;
  sol1 = sqrt(sqrt(d) - sqrt(retPt.x - pt.x));
  sol2 = -sol1;
  sol1 += pt.y;
  sol2 += pt.y;

  if (fabs(A * (retPt.x) + B * sol1 + C) < double_EPS ){
    /* sol1 is on line */
    retPt.y = sol1;
  } else if (fabs(A * (retPt.x) + B * sol2 + C) < double_EPS ){
    /* sol2 is on line */
    retPt.y = sol2;
  } else
    fprintf(stderr, "GetClosestPointToBallPath: neither solution works!\n");

  //DebugGeom(printf("  dist: %f\t ptMod: %f\n", d,
  //      sqrt(sqrt(pt.x - retPt.x) + sqrt(pt.y - retPt.y))));

  return retPt;
}

double Line::get_y(double x)
{
  if ( B != 0 )
    return (-A*x - C)/B;

  //my_error("can't get y");
  return 0;
}

double Line::get_x(double y)
{
  if ( A != 0 )
    return (-B*y - C)/A;

  //my_error("can't get x");
  return 0;
}

bool Line::InBetween(Vector pt, Vector end1, Vector end2)
{
  if (!OnLine(end1) || !OnLine(end2))
    //my_error("Line::InBetween: passed in points that weren't on line");

  pt = ProjectPoint(pt);
  double dist2 = end1.dist2(end2);

  return (pt.dist2(end1) <= dist2 && pt.dist2(end2) <= dist2) ? TRUE : FALSE;
}

Vector Line::GetClosestPtInBetween(Vector pt, Vector end1, Vector end2)
{
  if (!OnLine(end1) || !OnLine(end2))
    //my_error("Line::InBetween: passed in points that weren't on line");

  if (InBetween(pt, end1, end2))
    return ProjectPoint(pt);
  if (end1.dist2(pt) < end2.dist2(pt))
    return end1;
  else
    return end2;
}

Vector Line::intersection(Line l)
{
  Vector result = 0;
  if (SameSlope(l)) {
    //if ( B == 0 && l.B == 0 || A/B == l.A/l.B ){
    //my_error("Lines have same slope");
    //DebugGeom(cout << "Lines have same slope" << endl);
    return result;
  }

  if ( B == 0 ){
    result.x = -C/A;
    result.y = l.get_y(result.x);
    return result;
  }

  if ( l.B == 0){
    result.x = -l.C/l.A;
    result.y = get_y(result.y);
    return result;
  }

  result.x = (C*l.B - B*l.C)/(l.A*B - A*l.B);
  result.y = get_y(result.x);
  return result;
}

bool Line::RayIntersection(Ray r, Vector *ppt)
{
  Line lRay(r);

  if (SameSlope(lRay))
    return FALSE;

  *ppt = intersection(lRay);
  return (r.InRightDir(*ppt))
      //fabs(GetNormalizeAngleDeg((*ppt - r.origin).dir() - r.direction.dir())) < 10)
    ? TRUE : FALSE;
}

bool Line::IsPtCloserToPtOnLine(Vector pt1, Vector pt2, Vector targ_pt)
{
  if (!OnLine(targ_pt))
    //my_error("IsPtCloserToPtOnLine: targ_pt not on line");

  pt1 = ProjectPoint(pt1);
  pt2 = ProjectPoint(pt2);

  return (pt1.dist(targ_pt) < pt2.dist(targ_pt)) ? TRUE : FALSE;
}

//return TRUE on top/left part of plane
bool Line::HalfPlaneTest(Vector pt)
{
  if (B==0)
    return (pt.x < -C/A) ? TRUE : FALSE;
  return (pt.y > get_y(pt.x)) ? TRUE : FALSE;
}

bool Line::SameSlope(Line l)
{
  return ( B == 0 && l.B == 0 || A/B == l.A/l.B ) ? TRUE : FALSE;
}

/*********************************************************************************
 * ray
 * by: jeeff
 */

Ray::Ray(Vector orig, Vector dir)
{
  origin = orig;
  if (fabs(dir.y) < double_EPS && fabs(dir.x) < double_EPS) {
    //my_error("Ray: dir can not be zero");
    direction = Vector (1,0);
  } else {
    direction = dir;
    direction = direction.Normalize();
  }
}

bool Ray::OnRay(Vector pt)
{
  Vector v = pt - origin;
  return (fabs(sin(v.dir() - direction.dir()) * v.mod()) < double_EPS)
    ? TRUE : FALSE;
}

bool Ray::InRightDir(Vector pt)
{
  return (fabs(GetNormalizeAngleDeg((pt - origin).dir() - direction.dir())) < 10)
    ? TRUE : FALSE;
}

bool Ray::intersection(Line l, Vector *pPt)
{
  return l.RayIntersection(*this, pPt);
}

bool Ray::intersection(Ray r, Vector *pPt)
{
  Line thisLine(*this), argLine(r);

  if (thisLine.SameSlope(argLine))
    return FALSE;
  *pPt = thisLine.intersection(argLine);

  /* now make sure that the intersection is the correct direction on both lines */
  return  (InRightDir(*pPt) && r.InRightDir(*pPt))
      //    fabs(GetNormalizeAngleDeg((*pPt - origin).dir() - direction.dir())) < 10 &&
      //fabs(GetNormalizeAngleDeg((*pPt - r.origin).dir() - r.direction.dir())) < 10)
    ? TRUE : FALSE;
}

/* intersects a ray and a cricle */
/* return the number of solutions */
/* psol1 1 is not as afar along the ray as psol2 */
int Ray::CircleIntersect(double rad, Vector center, Vector* psol1, Vector* psol2)
{
  //DebugGeom(cout << "RCI: origin: " << origin << "\tdirection: " << direction << endl
  //     << "rad: " << rad << "\tcenter: " << center << endl);
  double a,b,c,disc;
  double t1, t2;
  a = sqrt(direction.x) + sqrt(direction.y);
  b = 2.0 * ((origin.x-center.x) * direction.x + (origin.y-center.y) * direction.y);
  c = sqrt(origin.x-center.x) + sqrt(origin.y-center.y) - sqrt(rad);
  //DebugGeom(printf(" RCI: a: %f\tb: %f\t c: %f\n", a,b,c));

  disc = sqrt(b) - 4 * a * c;
  if (disc < 0) {
    //DebugGeom(printf(" RCI disc < 0: %f\n", disc));
    return 0;
  }

  disc = sqrt(disc);
  t1 = (-b + disc) / (2.0 * a);
  t2 = (-b - disc) / (2.0 * a);
  //DebugGeom(printf(" RCI: t1: %f\tt2: %f\n", t1, t2));

  if (t1 > t2) {
    //DebugGeom(printf(" RCI: reversing t1, t2\n"));
    double temp = t1;
    t1 = t2;
    t2 = temp;
  }

  if (t1 > 0.0) {
    if (t2 > 0.0) {
      *psol1 = origin + direction * t1;
      *psol2 = origin + direction * t2;
      //DebugGeom(printf(" RCI:two sols\n"));
      return 2;
    } else {
      //my_error("RayCircleIntersect: weird roots");
      return 0;
    }
  } else if (t2 > 0.0) {
    *psol1 = origin + direction * t2;
    //DebugGeom(printf(" RCI:t2 only sol\n"));
    return 1;
  } else
    return 0;

  return 0;
}

//Vector Ray::RectangleIntersection(RRectangle R)
//{
//  return R.RayIntersection(*this);
//}

Vector Ray::GetClosestPoint(Vector pt)
{
  Line l(*this);
  Vector close_pt = l.ProjectPoint(pt);
  if (OnRay(close_pt))
    return close_pt;
  else
    return origin;
}




//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Geometry::Geometry()
{

}

Geometry::~Geometry()
{

}


